<?php
/**
 * Extract N Word
 *
 * @package     AutomatorWP\Integrations\Formatter\Actions\Extract_N_Word
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly

if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Formatter_Extract_N_Word extends AutomatorWP_Integration_Action {

    public $integration = 'formatter';
    public $action = 'formatter_extract_n_word';
    public $formatted = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Extract the word n from a string', 'automatorwp-pro' ),
            'select_option'     => __( 'Extract <strong>the word n</strong> from a <strong>string</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Position. %2$s: String. */
            'edit_label'        => sprintf( __( 'Extract the word %1$s from %2$s', 'automatorwp-pro' ), '{position}', '{string}' ),
            /* translators: %1$s: Position. */
            'log_label'         => sprintf( __( 'Extract the word %1$s from a %2$s', 'automatorwp-pro' ), '{position}', '{string}' ),
            'options'           => array(
                'string' => array(
                    'default' => __( 'string', 'automatorwp-pro' ),
                    'fields' => array(
                        'string' => array(
                            'name' => __( 'String:', 'automatorwp-pro' ),
                            'desc' => __( 'String to extract the first word.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required' => true,
                            'default' => ''
                        ),
                     ) ),
                'position' => array(
                    'from'  => 'position',
                    'default' => __( 'n', 'automatorwp-pro' ),
                    'fields' => array(
                        'position' => array(
                            'name' => __( 'Position:', 'automatorwp-pro' ),
                            'desc' => __( 'Position of the word in the string.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required' => true,
                            'default' => ''
                        ),
                    ) ),
            ),
            'tags'  => automatorwp_formatter_get_actions_string_tags()
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';
        
        // Shorthand
        $string = $action_options['string'];
        $position = $action_options['position'] - 1;

        // Bail if not string
        if ( empty( $string ) ) {
            $this->result = __( 'Please, insert a string to be formatted', 'automatorwp-pro' );
            return;
        }

        // Bail if not position
        if ( empty( $position ) ) {
            $this->result = __( 'Please, insert the position of the word to extract', 'automatorwp-pro' );
            return;
        }

        // Get word in N position
        $words_array = explode(' ', trim( $string ) );

        if ( !isset( $words_array[$position] ) ) {
            $this->result = sprintf( __( 'The position could not be found. The string has %d words', 'automatorwp-pro' ), count( $words_array ) );
            return;
        }

        $this->formatted = $words_array[$position];

        $this->result = sprintf( __( '%s', 'automatorwp-pro' ), $this->formatted );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;
        $log_meta['formatted'] = ( isset( $this->formatted ) ? $this->formatted : '' );

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Formatter_Extract_N_Word();